package com.hikcreate.ftp.proxy.base.exception;

import com.hikcreate.ftp.proxy.base.response.Response;
import com.hikcreate.ftp.proxy.base.response.ResponseGenerator;
import com.hikcreate.ftp.proxy.base.statuscode.StatusCode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;

/**
 * 统一异常处理
 *
 * @author zhangfubing
 */
@Slf4j
@ControllerAdvice
public class DefaultExceptionHandler {

    @ExceptionHandler(value = BindException.class)
    @ResponseBody
    public Response<Void> handle(BindException e) {
        BindingResult bindingResult = e.getBindingResult();
        String errorMessage = buildErrorMessage(bindingResult);
        log.error(errorMessage);

        FieldError fieldError = e.getFieldError();
        String message = fieldError.getDefaultMessage();

        return ResponseGenerator.fail(StatusCode.PARAM_ERROR.getCode(), message, errorMessage);
    }

    /**
     * 可能原因：@Validated 注解的对象验证不通过
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseBody
    public Response<Void> handle(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String errorMessage = buildErrorMessage(bindingResult);
        log.error(errorMessage);
        List<FieldError> fieldErrors = bindingResult.getFieldErrors();
        String message = "参数错误";
        if (null != fieldErrors && fieldErrors.size() > 0) {
            FieldError fieldError = fieldErrors.get(0);
            if (null != fieldError) {
                message = fieldError.getDefaultMessage();
            }
        }
        return ResponseGenerator.fail(StatusCode.PARAM_ERROR.getCode(), message, errorMessage);
    }

    /**
     * 服务类异常
     *
     * @param e 异常对象
     * @return 接口返回对象
     */
    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public Response<Void> handle(BusinessException e) {
        log.error(e.toString(), e);
        return ResponseGenerator.fail(e.getCode(), e.getMsg(), e.getErrorMsg());
    }

    /**
     * 其他异常
     *
     * @param e 异常对象
     * @return 接口返回对象
     */
    @ExceptionHandler(Throwable.class)
    @ResponseBody
    public Response<Void> handle(Throwable e) {
        log.error(e.toString(), e);
        return ResponseGenerator.fail(StatusCode.SYSTEM_ERROR, getStackTrace(e));
    }

    /**
     * 从{@code @BindingResult}中构建异常信息
     * on field 'verifyCode': null];
     *
     * @param bindingResult
     */
    private String buildErrorMessage(BindingResult bindingResult) {

        StringBuffer sb = new StringBuffer("BindException. ");
        sb.append("Field error in object '" + bindingResult.getObjectName() + "'. [" + bindingResult.getTarget() + "]");
        bindingResult.getFieldErrors()
                .forEach(error -> {
                    sb.append("\r\n on field '" + error.getField() + "': ");
                    sb.append("rejected value [" + error.getRejectedValue() + "]. ");
                    sb.append("default message [" + error.getDefaultMessage() + "]");
                });
        return sb.toString();
    }

    /**
     * 获取异常对象的堆栈信息
     *
     * @param throwable 异常对象
     * @return 堆栈信息
     */
    protected String getStackTrace(Throwable throwable) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw)) {
            throwable.printStackTrace(pw);
            return sw.toString();
        }
    }
}
