package com.hikcreate.edl.pub.web.mobile.domain.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.hikcreate.common.orika.DefaultConverter;
import com.hikcreate.common.sdk.exception.BusinessException;
import com.hikcreate.common.sdk.response.apiparam.Response;
import com.hikcreate.common.sdk.response.statuscode.StatusCode;
import com.hikcreate.edl.common.distributed.lock.annotation.DistributedLock;
import com.hikcreate.edl.common.sdk.util.VerifyCodeUtil;
import com.hikcreate.edl.pub.web.mobile.domain.IBindService;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.Result;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.ResultCode;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.ResultGenerator;
import com.hikcreate.edl.pub.web.mobile.infra.core.util.UnqIdUtil;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.BindCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.DrivingLicenseCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.VechicleCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.ViolationCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.IdCardQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumAndTypeQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.DrivingLicenseRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.VehicleRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.SmsFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.param.request.CheckVerifyCodeMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.param.request.SendVerifyCodeMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.DrivingLicenseFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.request.IdCardMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.response.DlQrRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.pvt_service_traffic.parm.request.ViolationListByTimeReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.pvt_service_traffic.parm.response.ViolationMicRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.BindInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.VerifyInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.model.entity.ParkBindInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.entity.ParkVerifyInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.request.*;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.BindInfoQueryRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.BindInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.InsuranceInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ResultList;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.VehicleDrivingLicenseInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ViolationInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.YearCheckInfoRes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 客户绑定服务实现类
 *
 * @author: xieshixiang
 * @time：2020/6/18 16:12
 **/
@Service
@Slf4j
public class BindServiceImpl extends ServiceImpl<BindInfoMapper, ParkBindInfo> implements IBindService {

    @Resource
    private BindInfoMapper mapper;
    @Resource
    private VerifyInfoMapper verifyInfo;
    @Resource
    private BindCache bindCache;
    @Resource
    private VechicleCache vechicleCache;
    @Resource
    private ViolationCache violationCache;
    @Resource
    private DrivingLicenseCache drivingLicenseCache;
    @Autowired
    private UnqIdUtil unqIdUtil;
    @Autowired
    private DrivingLicenseFeign drivingLicenseFeign;
    @Autowired
    private SmsFeign smsFeign;
    @Autowired
    DefaultConverter defaultConverter;


    /**
     * 解除绑定时间限制
     * 单位：小时
     */
    @Value("${biz.unbind.time:720}")
    private Integer unBindTime;
    /**
     * 解绑限制次数
     */
    @Value("${biz.unbind.limit.count:10}")
    private Integer unBindCount;

    /**
     * 验证有效性限制时间
     * 单位：小时
     */
    @Value("${biz.check.validity.limit.time:24}")
    private Integer checkValidityLimitTime;

    @Autowired
    private StringRedisTemplate redisTemplate;

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final String bindKeyPrefix = "park:extend:bindInfo:";


    @Override
    @DistributedLock(key = "'park:extend:lock:bind:userId:'+#bindInfoReq.userId+':timestamp:'+#bindInfoReq.timestamp", expireTime =
            3000, waitTime = 2000, retryTimes = 50)
    public Result bind(BindInfoReq bindInfoReq) {
        ParkBindInfo info = defaultConverter.getMapperFacade().map(bindInfoReq, ParkBindInfo.class);
        //判断规则1:用户已绑定的车辆不超过3辆,包括已解绑但未过一个月的车辆
        QueryWrapper<ParkBindInfo> boundQuery = new QueryWrapper();
        boundQuery.eq("user_id", info.getUserId());
        boundQuery.eq("status", "1");
        List<ParkBindInfo> list = mapper.selectList(boundQuery);
        if (list != null && list.size() >= 3) {
            return ResultGenerator.fail(ResultCode.BIND_NUM_ERROR);
        }
        /**
         * 规则:30天内,同一客户解绑超过10次不能再进行绑定
         */
        QueryWrapper<ParkBindInfo> unbindQuery = new QueryWrapper<>();
        unbindQuery.eq("user_id", info.getUserId());
        unbindQuery.ge("unbind_time", DateUtil.offsetHour(new Date(), -unBindTime));
        unbindQuery.eq("status", "0");
        Integer integer = mapper.selectCount(unbindQuery);
        if (integer >= unBindCount) {
            return ResultGenerator.fail(ResultCode.UNBIND_TOTAL_ERROR);
        }

        //规则:一个客户只能绑定一个电话号码
        if (list != null && !list.isEmpty()) {
            boolean equals = list.get(0).getPhone().equals(info.getPhone());
            if (!equals) {
                return ResultGenerator.fail(ResultCode.USERID_PHONE_ERROR);
            }
        }

        //规则:一个电话号码只能被一个客户绑定
        QueryWrapper<ParkBindInfo> phone = new QueryWrapper();
        phone.eq("phone", info.getPhone());
        phone.eq("status", "1");
        List<ParkBindInfo> phones = mapper.selectList(phone);
        if (phones != null && !phones.isEmpty()) {
            boolean phoneExist = phones.get(0).getUserId().equals(info.getUserId());
            if (!phoneExist) {
                return ResultGenerator.fail(StatusCode.DATA_ERROR.getCode(), "当前手机号已被其他客户绑定");
            }

        }

        //规则2:客户手机号与车辆对应的车主手机号一致
        PlateNumQueryReq req = new PlateNumQueryReq();
        req.setPlateNum(info.getPlateNum());
        List<VehicleRes> byPlateNum = vechicleCache.getByPlateNum(req);
        if (byPlateNum == null || byPlateNum.size() == 0) {
            return ResultGenerator.fail(ResultCode.CAR_MESSAGE_UNKNOWN);
        }

        for (VehicleRes res : byPlateNum) {
            if (info.getOwnerPhone().equals(res.getSjhm())) {
                info.setPlateType(res.getHpzl());
            }
        }
        if (info.getPlateType() == null || "".equals(info.getPlateType())) {
            return ResultGenerator.fail(ResultCode.CAR_PHONE_UNMATCHED);
        }

        //规则3:判断当前车牌号是否已绑定
        QueryWrapper<ParkBindInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", info.getUserId());
        queryWrapper.eq("plate_num", info.getPlateNum());
        queryWrapper.eq("plate_type", info.getPlateType());
        queryWrapper.eq("status", "1");
        ParkBindInfo bindExist = mapper.selectOne(queryWrapper);
        if (bindExist != null) {
            return ResultGenerator.fail(ResultCode.CAR_HAVE_BIND);
        }

        //保存至redis
        try {
            String key = bindKeyPrefix.concat(info.getPhone()).concat("-").concat(info.getOwnerPhone());
            redisTemplate.opsForValue().set(key, objectMapper.writeValueAsString(info), 30, TimeUnit.MINUTES);
        } catch (Exception e) {
            log.error("绑定异常：{}", e.getMessage());
            return ResultGenerator.fail(StatusCode.SYSTEM_ERROR);
        }
        //发送验证码
        SendVerifyCodeMicReq sendMicReq = new SendVerifyCodeMicReq();
        sendMicReq.setPhone(info.getOwnerPhone());
        sendMicReq.setVerifyCode(VerifyCodeUtil.getCode());
        sendMicReq.setTemplateCode("PARK_BIND_VEHICLE");
        smsFeign.sendVerifyCode(sendMicReq).fallback();

        return ResultGenerator.success();
    }

    /**
     * 车辆绑定短信验证接口
     *
     * @param req
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @DistributedLock(key = "'park:extend:lock:bind:smsValid:phone:'+#req.phone+':ownerPhone:'+#req.ownerPhone", isSpin = false)
    public Result bindSmsValid(BindSmsValidReq req) {
        try {
            //获取缓存信息
            String key = bindKeyPrefix.concat(req.getPhone()).concat("-").concat(req.getOwnerPhone());
            String bindInfoStr = redisTemplate.opsForValue().get(key);
            if (StrUtil.isBlank(bindInfoStr)) {
                log.error("车辆绑定短信验证失败：未发起车辆绑定");
                return ResultGenerator.fail(StatusCode.BUSINESS_ERROR.getCode(), "已超时");
            }
            ParkBindInfo bindInfo = objectMapper.readValue(bindInfoStr, ParkBindInfo.class);

            //验证短信
            CheckVerifyCodeMicReq checkVerifyCodeMicReq = new CheckVerifyCodeMicReq();
            checkVerifyCodeMicReq.setPhone(req.getOwnerPhone())
                    .setVerifyCode(req.getSmsCode()).setTemplateCode("PARK_BIND_VEHICLE");
            smsFeign.checkVerifyCode(checkVerifyCodeMicReq).fallback();

            //绑定信息入库
            bindInfo.setUnqId(unqIdUtil.getUnqId());
            bindInfo.setGmtBindTime(new Date());
            if (!bindCache.insert(bindInfo)) {
                log.error("车辆绑定短信验证失败：绑定信息入库失败");
                return ResultGenerator.fail(StatusCode.SYSTEM_ERROR);
            }
            return ResultGenerator.success(new BindInfoRes(bindInfo.getUnqId()));
        } catch (JsonProcessingException e) {
            log.error("车辆绑定短信验证异常：{}", e.getMessage());
            return ResultGenerator.fail(StatusCode.SYSTEM_ERROR);
        }
    }

    /**
     * 信息解绑
     *
     * @param req
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @DistributedLock(key = "'park:extend:lock:unbind:unqId:'+#req.unqId+':userId:'+#req.userId", isSpin = false)
    public Result unbindInfo(UnBindReq req) {
        ParkBindInfo bindInfo = bindCache.getById(req.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus()) || !req.getUserId().equals(bindInfo.getUserId())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }
        if (!bindCache.unBind(req.getUnqId(), req.getUserId())) {
            //手动回滚
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultGenerator.fail("解绑失败");
        }
        return ResultGenerator.success();
    }

    /**
     * 车辆解绑发送短信(后台操作)
     *
     * @param req
     * @return
     */
    @Override
    @DistributedLock(key = "'park:extend:lock:adnmUnbind:unqId:'+#req.unqId", isSpin = false)
    public Result adminUnbind(AdminUnbindReq req) {
        //获取绑定信息
        ParkBindInfo parkBindInfo = bindCache.getById(req.getUnqId());
        if (parkBindInfo == null || "0".equals(parkBindInfo.getStatus())
                || !parkBindInfo.getUserId().equals(req.getUserId())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }
        //发送验证码
        SendVerifyCodeMicReq sendMicReq = new SendVerifyCodeMicReq();
        sendMicReq.setPhone(parkBindInfo.getOwnerPhone());
        sendMicReq.setVerifyCode(VerifyCodeUtil.getCode());
        sendMicReq.setTemplateCode("PARK_UNBIND_VEHICLE");
        smsFeign.sendVerifyCode(sendMicReq).fallback();

        return ResultGenerator.success();
    }

    /**
     * 车辆解绑短信验证（后台操作）
     *
     * @param req
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @DistributedLock(key = "'park:extend:lock:adminUnbind:smsValid:unqId:'+#req.unqId", isSpin = false)
    public Result adminUnbindSmsValid(AdminUnbindSmsValidReq req) {
        //获取绑定信息
        ParkBindInfo parkBindInfo = bindCache.getById(req.getUnqId());
        if (parkBindInfo == null || "0".equals(parkBindInfo.getStatus())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }
        //验证短信
        CheckVerifyCodeMicReq checkVerifyCodeMicReq = new CheckVerifyCodeMicReq();
        checkVerifyCodeMicReq.setPhone(parkBindInfo.getOwnerPhone())
                .setVerifyCode(req.getSmsCode()).setTemplateCode("PARK_UNBIND_VEHICLE");
        smsFeign.checkVerifyCode(checkVerifyCodeMicReq).fallback();

        //解绑
        if (!bindCache.unBind(parkBindInfo.getUnqId(), parkBindInfo.getUserId())) {
            //手动回滚
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultGenerator.fail("解绑失败");
        }
        return ResultGenerator.success();
    }

    /**
     * 查询用户绑定信息
     *
     * @param query
     * @return
     */
    @Override
    public List<BindInfoQueryRes> selectByQuery(BindInfoQueryReq query) {
        List<ParkBindInfo> parkBindInfos = bindCache.selectByQuery(query.getUserId());
        List<BindInfoQueryRes> resList = new ArrayList<>(3);
        if (parkBindInfos != null && !parkBindInfos.isEmpty()) {
            for (ParkBindInfo item : parkBindInfos) {
                BindInfoQueryRes resultItem = new BindInfoQueryRes();
                resultItem.setPlateColor(item.getPlateColor());
                resultItem.setPlateNum(item.getPlateNum());
                resultItem.setUnqId(item.getUnqId());
                resultItem.setGmtBindTime(item.getGmtBindTime().getTime());
                resList.add(resultItem);
            }

        }
        return resList;
    }

    @Override
    public Result<YearCheckInfoRes> yearCheckInfo(H5BindInfoQuery query) {
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }
        if (!bindInfo.getUserId().equals(query.getUserId()) || !bindInfo.getPhone().equals(query.getPhone())) {
            return ResultGenerator.fail(StatusCode.PARAM_ERROR.getCode(), "客户唯一编码或手机号不正确");
        }

        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicleCache.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResultGenerator.success();
        }
        YearCheckInfoRes yearCheckInfoRes = new YearCheckInfoRes();
        yearCheckInfoRes.setPlateNum(bindInfo.getPlateNum());
        if (StrUtil.isNotBlank(vehicleRes.getYxqz())) {
            yearCheckInfoRes.setValidityTime(vehicleRes.getYxqz().substring(0, 10));
            Long checkTime = DateUtil.parse(vehicleRes.getYxqz(), "yyyy-MM-dd HH:mm:ss").getTime();
            if (checkTime >= System.currentTimeMillis()) {
                yearCheckInfoRes.setStatus("未到期");
            } else {
                yearCheckInfoRes.setStatus("已到期");
            }
        }
        return ResultGenerator.success(yearCheckInfoRes);
    }

    @Override
    public Result<InsuranceInfoRes> insuranceInfo(H5BindInfoQuery query) {
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }

        if (!bindInfo.getUserId().equals(query.getUserId()) || !bindInfo.getPhone().equals(query.getPhone())) {
            return ResultGenerator.fail(StatusCode.PARAM_ERROR.getCode(), "客户唯一编码或手机号不正确");
        }

        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicleCache.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResultGenerator.success();
        }
        InsuranceInfoRes insuranceInfoRes = new InsuranceInfoRes();
        insuranceInfoRes.setPlateNum(bindInfo.getPlateNum());
        insuranceInfoRes.setInsuranceType("交强险");
        if (StrUtil.isNotBlank(vehicleRes.getBxzzrq()) && !"null".equalsIgnoreCase(vehicleRes.getBxzzrq())) {
            Long time = DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss").getTime();
            if (time >= System.currentTimeMillis()) {
                insuranceInfoRes.setStatus("有效");
            } else {
                insuranceInfoRes.setStatus("已到期");
            }
            insuranceInfoRes.setValidityTimeEnd(vehicleRes.getBxzzrq().substring(0, 10));
            String endTime = DateUtil.format(
                    DateUtil.offsetMonth(DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss"), -12),
                    "yyyy-MM-dd");
            insuranceInfoRes.setValidityTimeStart(endTime);
        } else {
            insuranceInfoRes.setStatus("未知");
        }
        return ResultGenerator.success(insuranceInfoRes);
    }


    @Override
    public Result<ResultList<ViolationInfoRes>> violationInfo(H5BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }

        if (!bindInfo.getUserId().equals(query.getUserId()) || !bindInfo.getPhone().equals(query.getPhone())) {
            return ResultGenerator.fail(StatusCode.PARAM_ERROR.getCode(), "客户唯一编码或手机号不正确");
        }

        /**
         * 查询违章信息
         */
        ViolationListByTimeReq queryReq = new ViolationListByTimeReq();
        queryReq.setPlateNum(bindInfo.getPlateNum());
        queryReq.setPlateType(bindInfo.getPlateType());
        queryReq.setTimeEnd(new Date());
        queryReq.setTimeStart(DateUtil.offsetMonth(new Date(), -12));
        List<ViolationMicRes> byPlateNumAndType = violationCache.getByPlateNumAndType(queryReq);
        //过滤已处理信息
        byPlateNumAndType = byPlateNumAndType.stream().filter(v -> !"1".equals(v.getStatus())).collect(Collectors.toList());

        /**
         * 构建违章信息返回列表
         */
        List<ViolationInfoRes> violationInfoResList = new ArrayList<>();
        for (ViolationMicRes res : byPlateNumAndType) {
            ViolationInfoRes info = new ViolationInfoRes();
            info.setPlateNum(res.getPlateNum());
            info.setAction(res.getVioDetailName());
            info.setAddress(res.getAddress());
            info.setGrade(res.getScore().toString());
            info.setMoney(res.getForfeit().toString());
            info.setTime(res.getDate());
            violationInfoResList.add(info);
        }
        return ResultGenerator.success(new ResultList<>(violationInfoResList));
    }

    @Override
    public Result<VehicleDrivingLicenseInfoRes> drivingLicenseInfo(H5BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResultGenerator.fail(ResultCode.BIND_INFO_UNKNOWN);
        }
        if (!bindInfo.getUserId().equals(query.getUserId()) || !bindInfo.getPhone().equals(query.getPhone())) {
            return ResultGenerator.fail(StatusCode.PARAM_ERROR.getCode(), "客户唯一编码或手机号不正确");
        }

        /**
         * 校验当前电话号码是否在1天内已通过验证
         */
        ParkVerifyInfo verify = verifyInfo.checkValidity(query.getPhone(), query.getUserId(),
                DateUtil.offsetHour(DateTime.now(), -checkValidityLimitTime));
        if (verify == null) {
            return ResultGenerator.fail(StatusCode.PERMISSION_DENIED);
        }

        /**
         * 根据车牌号和车牌类型查询车辆信息
         */
        PlateNumAndTypeQueryReq vechicleInfoReq = new PlateNumAndTypeQueryReq();
        vechicleInfoReq.setPlateType(bindInfo.getPlateType());
        vechicleInfoReq.setPlateNum(bindInfo.getPlateNum());
        VehicleRes byPlateNumAndType = vechicleCache.getByPlateNumAndType(vechicleInfoReq);
        if (byPlateNumAndType.getSfzmhm() == null) {
            return ResultGenerator.fail(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }

        /**
         * 根据身份证号码查询驾照信息
         */
        IdCardQueryReq idCardQueryReq = new IdCardQueryReq();
        idCardQueryReq.setIdCard(byPlateNumAndType.getSfzmhm());
        DrivingLicenseRes drivingLicenseRes = drivingLicenseCache.getByIdCard(idCardQueryReq);
        if (drivingLicenseRes == null) {
            return ResultGenerator.fail(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }

        /**
         * 获取二维码信息
         */
        Response<DlQrRes> qrInfo = drivingLicenseFeign.getQrInfo(new IdCardMicReq().setIdCard(byPlateNumAndType.getSfzmhm()));
        if (!qrInfo.isSuccess() || qrInfo.getData() == null) {
            return ResultGenerator.fail(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }
        /**
         * 构建驾照信息查询结果
         */
        VehicleDrivingLicenseInfoRes result = new VehicleDrivingLicenseInfoRes();
        result.setDriverName(drivingLicenseRes.getXm());
        result.setPlateNum(byPlateNumAndType.getHphm());
        result.setUserId(query.getUserId());
        result.setDlQr(qrInfo.getData().getQr());
        String validityTime = drivingLicenseRes.getYxqz();
        if (StrUtil.isNotBlank(validityTime) && !"null".equalsIgnoreCase(validityTime)) {
            validityTime = validityTime.substring(0, 10);
        }
        result.setValidityTime(validityTime);
        String status = DateUtil.compare(DateUtil.parseDate(drivingLicenseRes.getYxqz()), new Date()) >= 0 ? "有效" :
                "已过期";
        result.setStatus(status);
        result.setSurplusGrade(String.valueOf(12 - Integer.parseInt(drivingLicenseRes.getLjjf())));

        return ResultGenerator.success(result);
    }

    @Override
    public void saveValid(String userId, String unqId, String phone) {
        /**
         * 校验当前的绑定信息是否存在
         */
        ParkBindInfo bindInfo = bindCache.getById(unqId);
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            throw new BusinessException(ResultCode.BIND_INFO_UNKNOWN);
        }
        if (!bindInfo.getUserId().equals(userId) || !bindInfo.getPhone().equals(phone)) {
            throw new BusinessException(StatusCode.PARAM_ERROR.getCode(), "客户唯一编码或手机号不正确");
        }
        /**
         * 根据车牌号和车牌类型查询车辆信息
         */
        PlateNumAndTypeQueryReq vechicleInfoReq = new PlateNumAndTypeQueryReq();
        vechicleInfoReq.setPlateType(bindInfo.getPlateType());
        vechicleInfoReq.setPlateNum(bindInfo.getPlateNum());
        VehicleRes byPlateNumAndType = vechicleCache.getByPlateNumAndType(vechicleInfoReq);
        if (byPlateNumAndType.getSfzmhm() == null) {
            throw new BusinessException(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }

        /**
         * 根据身份证号码查询驾照信息
         */
        IdCardQueryReq idCardQueryReq = new IdCardQueryReq();
        idCardQueryReq.setIdCard(byPlateNumAndType.getSfzmhm());
        DrivingLicenseRes drivingLicenseRes = drivingLicenseCache.getByIdCard(idCardQueryReq);
        if (drivingLicenseRes == null) {
            throw new BusinessException(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }
        // 存至缓存，有效期一天
        redisTemplate.opsForValue().set(getValidKey(userId, drivingLicenseRes.getSfzmhm()), phone, 24 * 3600, TimeUnit.SECONDS);
    }

    /**
     * 验证是否已通过短信验证
     *
     * @param userId
     * @param idCard
     * @return
     */
    @Override
    public Boolean checkValid(String userId, String idCard) {
        String s = redisTemplate.opsForValue().get(getValidKey(userId, idCard));
        return !StringUtils.isEmpty(s);
    }

    private String getValidKey(String userId, String idCard) {
        return "VehValidKey:" + userId + ":" + idCard;
    }
}



