package com.hikcreate.edl.pub.web.mobile.domain.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.hikcreate.common.sdk.exception.BusinessException;
import com.hikcreate.edl.common.sdk.util.VerifyCodeUtil;
import com.hikcreate.edl.pub.web.mobile.domain.DrivingLicenseService;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.ResultCode;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.SmsFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.param.request.CheckVerifyCodeMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pub_service_sms.param.request.SendVerifyCodeMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.DrivingLicenseFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.request.CreditIdCardMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.request.IdCardMicReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.response.DlQrRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.edl_pvt_service_drivinglicense.response.DrivingLicenseMicRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.DrivingLicenseInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.DrivingLicenseValidRes;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;


/**
 * @author MOUBK
 * @create 2020/7/21 14:03
 */
@Slf4j
@Service
public class DrivingLicenseServiceImpl implements DrivingLicenseService {

    @Autowired
    private DrivingLicenseFeign drivingLicenseFeign;
    @Autowired
    private SmsFeign smsFeign;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;
    private static final String REDIS_KEY = "drv:idcard";
    private static final String REDIS_SMS_KEY = "drv:idcard:sms";

    @Override
    public DrivingLicenseValidRes checkIdCard(String idCard, String userId) {
        DrivingLicenseMicRes drivingLicenseMicRes = drivingLicenseFeign.getByIdCard(new CreditIdCardMicReq().setIdCard(idCard)).fallback().getData();
        if (drivingLicenseMicRes == null) {
            throw new BusinessException(ResultCode.DRIVING_LICENSE_UNKNOWN);
        }
        // 保存在缓存
        saveToRedis(userId, drivingLicenseMicRes);
        DrivingLicenseValidRes drivingLicenseValidRes = new DrivingLicenseValidRes();
        drivingLicenseValidRes.setUserId(userId);
        drivingLicenseValidRes.setPhone(drivingLicenseMicRes.getPhone());
        // 检查短信发送
        if (checkSmsValid(userId, idCard, drivingLicenseMicRes.getPhone())) {
            drivingLicenseValidRes.setSendSmsFlag(true);
        } else {
            drivingLicenseValidRes.setSendSmsFlag(false);
        }
        return drivingLicenseValidRes;
    }

    @Override
    public void sendSmsCode(String userId) {
        DrivingLicenseMicRes drivingLicenseMicRes = getFromRedis(userId);
        SendVerifyCodeMicReq micReq = new SendVerifyCodeMicReq();
        micReq.setPhone(drivingLicenseMicRes.getPhone());
        micReq.setVerifyCode(VerifyCodeUtil.getCode());
        micReq.setTemplateCode("PARK_DRIVING_LICENSE_IDCARD");
        smsFeign.sendVerifyCode(micReq).fallback();
    }

    @Override
    public void validSmsCode(String userId, String smsCode) {
        DrivingLicenseMicRes drivingLicenseMicRes = getFromRedis(userId);
        CheckVerifyCodeMicReq req = new CheckVerifyCodeMicReq();
        req.setPhone(drivingLicenseMicRes.getPhone());
        req.setVerifyCode(smsCode);
        req.setTemplateCode("PARK_DRIVING_LICENSE_IDCARD");
        smsFeign.checkVerifyCode(req).fallback();
        // 保存验证redis
        saveSmsValid(userId, drivingLicenseMicRes.getIdCard(), drivingLicenseMicRes.getPhone());
    }

    @Override
    public DrivingLicenseInfoRes getDrivingLicenseInfo(String userId) {
        DrivingLicenseMicRes drivingLicenseMicRes = getFromRedis(userId);
        DlQrRes data = drivingLicenseFeign.getQrInfo(new IdCardMicReq().setIdCard(drivingLicenseMicRes.getIdCard())).fallback().getData();
        /**
         * 构建驾照信息查询结果
         */
        DrivingLicenseInfoRes result = new DrivingLicenseInfoRes();
        result.setDriverName(drivingLicenseMicRes.getRealName());
        result.setDlQr(data.getQr());
        result.setValidityTime(drivingLicenseMicRes.getExpiryDate());
        String status = DateUtil.compare(DateUtil.parseDate(drivingLicenseMicRes.getExpiryDate()), new Date()) >= 0 ? "有效" :
                "已过期";
        result.setStatus(status);
        result.setUserId(userId);
        result.setIdCard(drivingLicenseMicRes.getIdCard());
        result.setSurplusGrade(String.valueOf(drivingLicenseMicRes.getResidualScore()));


        return result;
    }

    private void saveSmsValid(String userId, String idCard, String phone) {
        redisTemplate.opsForValue().set(getSmsKey(userId, idCard), phone, 300, TimeUnit.SECONDS);
    }

    private boolean checkSmsValid(String userId, String idCard, String phone) {
        String value = redisTemplate.opsForValue().get(getSmsKey(userId, idCard));
        if (Objects.equals(phone, value)) {
            return true;
        }
        return false;
    }

    @Override
    public Boolean checkValid(String userId, String idCard) {
        String value = redisTemplate.opsForValue().get(getSmsKey(userId, idCard));
        if (!StringUtils.isEmpty(value)) {
            return true;
        }
        return false;
    }

    private String getSmsKey(String userId, String idCard) {
        return REDIS_SMS_KEY + userId + ":" + idCard;
    }

    private void saveToRedis(String userId, DrivingLicenseMicRes drivingLicenseMicRes) {
        redisTemplate.opsForValue().set(getKey(userId), JSONObject.toJSONString(drivingLicenseMicRes), 24 * 3600, TimeUnit.SECONDS);
    }

    private DrivingLicenseMicRes getFromRedis(String userId) {
        String value = redisTemplate.opsForValue().get(getKey(userId));
        if (!StringUtils.isEmpty(value)) {
            return JSONObject.parseObject(value, DrivingLicenseMicRes.class);
        }
        return null;
    }

    private String getKey(String userId) {
        return REDIS_KEY + userId;
    }
}
