package com.hikcreate.edl.pub.web.mobile.domain.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.hikcreate.common.sdk.response.apiparam.Response;
import com.hikcreate.common.sdk.response.apiparam.ResponseGenerator;
import com.hikcreate.common.sdk.response.statuscode.StatusCode;
import com.hikcreate.edl.pub.web.mobile.domain.IBindService;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.DrivingLicenseFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.VechicleFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.ViolationFeign;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.IdCardQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumAndTypeQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.ViolationPlateNumAndTypeQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.DrivingLicenseRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.VehicleRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.ViolationRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.BindInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.VerifyInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.model.BindInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.BindInfoQuery;
import com.hikcreate.edl.pub.web.mobile.infra.model.VerifyInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.DrivingLicenseInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.InsuranceInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ResultList;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ViolationInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.YearCheckInfoRes;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * description:客户绑定服务实现类
 *
 * @author: xieshixiang
 * @time：2020/6/18 16:12
 **/
@Service
@EnableScheduling
@Slf4j
public class BindServiceImpl extends ServiceImpl<BindInfoMapper, BindInfo> implements IBindService {

    @Resource
    BindInfoMapper mapper;
    @Resource
    VechicleFeign vechicle;
    @Resource
    ViolationFeign violatio;
    @Resource
    VerifyInfoMapper verifyInfo;
    @Resource
    DrivingLicenseFeign drivingLicense;


    @Override
    public Response bind(BindInfo info) {
        info.setUnqId(UUID.randomUUID().toString());
        info.setGmtBindTime(DateTime.now().toString());

        //判断规则1:用户已绑定的车辆不超过3辆,包括已解绑但未过一个月的车辆
        QueryWrapper boundQuery = new QueryWrapper();
        boundQuery.eq("user_id", info.getUserId());
        Integer bound = mapper.selectCount(boundQuery);
        if (bound >= 3) {
            return ResponseGenerator.fail(StatusCode.BUSINESS_ERROR);
        }

        //规则2:客户手机号与车辆对应的车主手机号一致
        PlateNumQueryReq req = new PlateNumQueryReq();
        req.setPlateNum(info.getPlateNum());
        List<VehicleRes> byPlateNum = vechicle.getByPlateNum(req);
        for (VehicleRes res : byPlateNum) {
            if (info.getPhone().equals(res.getSjhm())) {
                info.setPlateType(res.getHpzl());
            }
        }
        if (info.getPlateType() == null || info.getPlateType().equals("")) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR);
        }
        return mapper.insert(info) > 0 ? ResponseGenerator.success(info.getUnqId()) :
                ResponseGenerator.fail(StatusCode.BUSINESS_ERROR);
    }

    @Override
    public Response<YearCheckInfoRes> yearCheckInfo(BindInfoQuery query) {
        BindInfo bindInfo = mapper.selectById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR);
        }
        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicle.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResponseGenerator.success();
        }
        YearCheckInfoRes yearCheckInfoRes = new YearCheckInfoRes();
        yearCheckInfoRes.setPlateNum(bindInfo.getPlateNum());
        yearCheckInfoRes.setValidityTime(vehicleRes.getYxqz());
        Long checkTime = DateUtil.parse(vehicleRes.getYxqz(), "yyyy-MM-dd HH:mm:ss").getTime();
        if (checkTime >= System.currentTimeMillis()) {
            yearCheckInfoRes.setStatus("未到期");
        } else {
            yearCheckInfoRes.setStatus("已到期");
        }
        return ResponseGenerator.success(yearCheckInfoRes);
    }

    @Override
    public Response<InsuranceInfoRes> insuranceInfo(BindInfoQuery query) {
        BindInfo bindInfo = mapper.selectById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR);
        }
        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicle.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResponseGenerator.success();
        }
        InsuranceInfoRes insuranceInfoRes = new InsuranceInfoRes();
        insuranceInfoRes.setPlateNum(bindInfo.getPlateNum());
        if (StrUtil.isNotBlank(vehicleRes.getBxzzrq())) {
            Long time = DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss").getTime();
            if (time >= System.currentTimeMillis()) {
                insuranceInfoRes.setStatus("有效");
            } else {
                insuranceInfoRes.setStatus("已到期");
            }
            insuranceInfoRes.setValidityTimeEnd(vehicleRes.getBxzzrq().substring(0, 10));
            String endTime = DateUtil.format(
                    DateUtil.offsetMonth(DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss"), -12),
                    "yyyy-MM-dd");
            insuranceInfoRes.setValidityTimeStart(endTime);
        } else {

        }
        return null;
    }


    @Override
    public Response<ResultList<ViolationInfoRes>> violationInfo(BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        BindInfo bindInfo = mapper.selectById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR);
        }

        /**
         * 查询违章信息
         */
        ViolationPlateNumAndTypeQueryReq req = new ViolationPlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        List<ViolationRes> byPlateNumAndType = violatio.getByPlateNumAndType(req);

        /**
         * 构建违章信息返回列表
         */
        List<ViolationInfoRes> violationInfoResList = new ArrayList<>();
        for (ViolationRes res : byPlateNumAndType) {
            ViolationInfoRes info = new ViolationInfoRes();
            info.setPlateNum(res.getHphm());
            info.setAction(res.getWfxw());
            info.setAddress(res.getWfdz());
            info.setGrade(res.getWfjfs());
            info.setMoney(res.getFkje());
            violationInfoResList.add(info);
        }
        return ResponseGenerator.success(new ResultList<>(violationInfoResList));
    }

    @Override
    public Response<DrivingLicenseInfoRes> drivingLicenseInfo(BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        BindInfo bindInfo = mapper.selectById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR);
        }

        /**
         * 校验当前电话号码是否在10天内已通过验证
         */
        VerifyInfo verifyInfo = this.verifyInfo.checkValidity(query.getPhone(), query.getUserId(),
                DateUtil.offsetDay(DateTime.now(), -10));
        if (verifyInfo == null) {
            return ResponseGenerator.fail(StatusCode.PERMISSION_DENIED);
        }

        /**
         * 根据车牌号和车牌类型查询车辆信息
         */
        PlateNumAndTypeQueryReq vechicleInfoReq = new PlateNumAndTypeQueryReq();
        vechicleInfoReq.setPlateType(bindInfo.getPlateType());
        vechicleInfoReq.setPlateNum(bindInfo.getPlateNum());
        VehicleRes byPlateNumAndType = vechicle.getByPlateNumAndType(vechicleInfoReq);
        if (byPlateNumAndType.getSfzmhm() == null) {
            return ResponseGenerator.fail(StatusCode.ALERT_ERROR);
        }

        /**
         * 根据身份证号码查询驾照信息
         */
        IdCardQueryReq idCardQueryReq = new IdCardQueryReq();
        idCardQueryReq.setIdCard(byPlateNumAndType.getSfzmhm());
        DrivingLicenseRes drivingLicenseRes = drivingLicense.getByIdCard(idCardQueryReq);
        if (drivingLicenseRes == null) {
            return ResponseGenerator.fail(StatusCode.ALERT_ERROR);
        }

        /**
         * 构建驾照信息查询结果
         */
        DrivingLicenseInfoRes result = new DrivingLicenseInfoRes();
        result.setDriverName(drivingLicenseRes.getXm());
        result.setPlateNum(drivingLicenseRes.getSfzmhm());
        result.setValidityTime(drivingLicenseRes.getYxqz());
        result.setStatus(drivingLicenseRes.getZtValue());

        return ResponseGenerator.success(result);
    }


    @Scheduled(cron = "0 0/30 * * * ? ")
    public void clearPast() {
        DateTime now = DateTime.now();
        log.info("开始清除,已过期且无效的绑定信息.当前时间:" + now);
        DateTime lastMonth = DateUtil.offsetDay(now, -30);
        Integer count = mapper.clearPast(lastMonth);
        log.info("清除完毕,共清除过期且无效的绑定信息:" + count);

    }


}
