package com.hikcreate.edl.pub.web.mobile.domain.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.hikcreate.common.sdk.response.apiparam.Response;
import com.hikcreate.common.sdk.response.apiparam.ResponseGenerator;
import com.hikcreate.common.sdk.response.statuscode.StatusCode;
import com.hikcreate.edl.pub.web.mobile.domain.IBindService;
import com.hikcreate.edl.pub.web.mobile.infra.core.util.UnqIdUtil;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.BindCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.DrivingLicenseCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.VechicleCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.cache.ViolationCache;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.IdCardQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumAndTypeQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.PlateNumQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.request.ViolationPlateNumAndTypeQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.DrivingLicenseRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.VehicleRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.feign.dcp_service_vehicle.param.response.ViolationRes;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.BindInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.data.mapper.VerifyInfoMapper;
import com.hikcreate.edl.pub.web.mobile.infra.model.entity.ParkBindInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.entity.ParkVerifyInfo;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.request.BindInfoQueryReq;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.request.H5BindInfoQuery;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.request.UnBindReq;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.BindInfoQueryRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.BindInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.DrivingLicenseInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.InsuranceInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ResultList;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.ViolationInfoRes;
import com.hikcreate.edl.pub.web.mobile.infra.model.param.response.YearCheckInfoRes;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 客户绑定服务实现类
 *
 * @author: xieshixiang
 * @time：2020/6/18 16:12
 **/
@Service
@Slf4j
public class BindServiceImpl extends ServiceImpl<BindInfoMapper, ParkBindInfo> implements IBindService {

    @Resource
    BindInfoMapper mapper;
    @Resource
    VerifyInfoMapper verifyInfo;
    @Resource
    BindCache bindCache;
    @Resource
    VechicleCache vechicleCache;
    @Resource
    ViolationCache violationCache;
    @Resource
    DrivingLicenseCache drivingLicenseCache;
    @Autowired
    UnqIdUtil unqIdUtil;


    @Override
    public Response bind(ParkBindInfo info) {
        //判断规则1:用户已绑定的车辆不超过3辆,包括已解绑但未过一个月的车辆
        QueryWrapper boundQuery = new QueryWrapper();
        boundQuery.eq("user_id", info.getUserId());
        boundQuery.eq("status", "1");
        Integer bound = mapper.selectCount(boundQuery);
        if (bound >= 3) {
            return ResponseGenerator.fail(StatusCode.BUSINESS_ERROR);
        }

        //规则2:客户手机号与车辆对应的车主手机号一致
        PlateNumQueryReq req = new PlateNumQueryReq();
        req.setPlateNum(info.getPlateNum());
        List<VehicleRes> byPlateNum = vechicleCache.getByPlateNum(req);
        if (byPlateNum == null || byPlateNum.size() == 0) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "车辆信息不存在");
        }

        for (VehicleRes res : byPlateNum) {
            if (info.getPhone().equals(res.getSjhm())) {
                info.setPlateType(res.getHpzl());
            }
        }
        if (info.getPlateType() == null || "".equals(info.getPlateType())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "当前手机号,与车辆备案手机号不一致");
        }

        //规则3:判断当前车牌号是否已绑定
        QueryWrapper<ParkBindInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", info.getUserId());
        queryWrapper.eq("plate_num", info.getPlateNum());
        queryWrapper.eq("plate_type", info.getPlateType());
        queryWrapper.eq("status", "1");
        ParkBindInfo bindExist = mapper.selectOne(queryWrapper);
        if (bindExist != null) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "此车牌号已绑定");
        }


        info.setUnqId(unqIdUtil.getUnqId());
        info.setGmtBindTime(DateTime.now().toString());
        mapper.insert(info);
        return ResponseGenerator.success(new BindInfoRes(info.getUnqId()));
    }

    /**
     * 信息解绑
     *
     * @param req
     * @return
     */
    @Override
    public boolean unbindInfo(UnBindReq req) {
        /**
         * 解绑规则:30天内,同一客户解绑次数总计不能超过10次
         */
        QueryWrapper<ParkBindInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", req.getUnqId());
        queryWrapper.eq("unbind_time", DateUtil.offsetDay(new Date(), -30));
        queryWrapper.eq("status", "0");

        Integer integer = mapper.selectCount(queryWrapper);
        if (integer >= 10) {
            return false;
        }

        return bindCache.unBind(req.getUnqId(), req.getUserId());

    }

    /**
     * 查询用户绑定信息
     *
     * @param query
     * @return
     */
    @Override
    public List<BindInfoQueryRes> selectByQuery(BindInfoQueryReq query) {
        return bindCache.selectByQuery(query.getUserId());
    }

    @Override
    public Response<YearCheckInfoRes> yearCheckInfo(H5BindInfoQuery query) {
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "绑定信息不存在");
        }
        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicleCache.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResponseGenerator.success();
        }
        YearCheckInfoRes yearCheckInfoRes = new YearCheckInfoRes();
        yearCheckInfoRes.setPlateNum(bindInfo.getPlateNum());
        if (StrUtil.isNotBlank(vehicleRes.getYxqz())) {
            yearCheckInfoRes.setValidityTime(vehicleRes.getYxqz().substring(0, 10));
            Long checkTime = DateUtil.parse(vehicleRes.getYxqz(), "yyyy-MM-dd HH:mm:ss").getTime();
            if (checkTime >= System.currentTimeMillis()) {
                yearCheckInfoRes.setStatus("未到期");
            } else {
                yearCheckInfoRes.setStatus("已到期");
            }
        }
        return ResponseGenerator.success(yearCheckInfoRes);
    }

    @Override
    public Response<InsuranceInfoRes> insuranceInfo(H5BindInfoQuery query) {
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "绑定信息不存在");
        }
        PlateNumAndTypeQueryReq req = new PlateNumAndTypeQueryReq();
        req.setPlateNum(bindInfo.getPlateNum());
        req.setPlateType(bindInfo.getPlateType());
        VehicleRes vehicleRes = vechicleCache.getByPlateNumAndType(req);
        if (Objects.isNull(vehicleRes)) {
            return ResponseGenerator.success();
        }
        InsuranceInfoRes insuranceInfoRes = new InsuranceInfoRes();
        insuranceInfoRes.setPlateNum(bindInfo.getPlateNum());
        insuranceInfoRes.setInsuranceType("交强险");
        if (StrUtil.isNotBlank(vehicleRes.getBxzzrq())) {
            Long time = DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss").getTime();
            if (time >= System.currentTimeMillis()) {
                insuranceInfoRes.setStatus("有效");
            } else {
                insuranceInfoRes.setStatus("已到期");
            }
            insuranceInfoRes.setValidityTimeEnd(vehicleRes.getBxzzrq().substring(0, 10));
            String endTime = DateUtil.format(
                    DateUtil.offsetMonth(DateUtil.parse(vehicleRes.getBxzzrq(), "yyyy-MM-dd HH:mm:ss"), -12),
                    "yyyy-MM-dd");
            insuranceInfoRes.setValidityTimeStart(endTime);
        } else {
            insuranceInfoRes.setStatus("未知");
        }
        return ResponseGenerator.success(insuranceInfoRes);
    }


    @Override
    public Response<ResultList<ViolationInfoRes>> violationInfo(H5BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "绑定信息不存在");
        }

        /**
         * 查询违章信息
         */
        ViolationPlateNumAndTypeQueryReq queryReq = new ViolationPlateNumAndTypeQueryReq();
        queryReq.setPlateNum(bindInfo.getPlateNum());
        queryReq.setPlateType(bindInfo.getPlateType());
        queryReq.setTimeEnd(new Date());
        queryReq.setTimeStart(DateUtil.offsetMonth(new Date(), -12));
        List<ViolationRes> byPlateNumAndType = violationCache.getByPlateNumAndType(queryReq);

        /**
         * 构建违章信息返回列表
         */
        List<ViolationInfoRes> violationInfoResList = new ArrayList<>();
        for (ViolationRes res : byPlateNumAndType) {
            ViolationInfoRes info = new ViolationInfoRes();
            info.setPlateNum(res.getHphm());
            info.setAction(res.getWfxw());
            info.setAddress(res.getWfdz());
            info.setGrade(res.getWfjfs());
            info.setMoney(res.getFkje());
            violationInfoResList.add(info);
        }
        return ResponseGenerator.success(new ResultList<>(violationInfoResList));
    }

    @Override
    public Response<DrivingLicenseInfoRes> drivingLicenseInfo(H5BindInfoQuery query) {
        /**
         * 校验当前的绑定信息是否存在
         */
        ParkBindInfo bindInfo = bindCache.getById(query.getUnqId());
        if (bindInfo == null || "0".equals(bindInfo.getStatus())) {
            return ResponseGenerator.fail(StatusCode.DATA_ERROR, "绑定信息不存在");
        }

        /**
         * 校验当前电话号码是否在10天内已通过验证
         */
        ParkVerifyInfo verify = verifyInfo.checkValidity(query.getPhone(), query.getUserId(),
                DateUtil.offsetDay(DateTime.now(), -10));
        if (verify == null) {
            return ResponseGenerator.fail(StatusCode.PERMISSION_DENIED);
        }

        /**
         * 根据车牌号和车牌类型查询车辆信息
         */
        PlateNumAndTypeQueryReq vechicleInfoReq = new PlateNumAndTypeQueryReq();
        vechicleInfoReq.setPlateType(bindInfo.getPlateType());
        vechicleInfoReq.setPlateNum(bindInfo.getPlateNum());
        VehicleRes byPlateNumAndType = vechicleCache.getByPlateNumAndType(vechicleInfoReq);
        if (byPlateNumAndType.getSfzmhm() == null) {
            return ResponseGenerator.fail(StatusCode.ALERT_ERROR);
        }

        /**
         * 根据身份证号码查询驾照信息
         */
        IdCardQueryReq idCardQueryReq = new IdCardQueryReq();
        idCardQueryReq.setIdCard(byPlateNumAndType.getSfzmhm());
        DrivingLicenseRes drivingLicenseRes = drivingLicenseCache.getByIdCard(idCardQueryReq);
        if (drivingLicenseRes == null) {
            return ResponseGenerator.fail(StatusCode.ALERT_ERROR);
        }

        /**
         * 构建驾照信息查询结果
         */
        DrivingLicenseInfoRes result = new DrivingLicenseInfoRes();
        result.setDriverName(drivingLicenseRes.getXm());
        result.setPlateNum(drivingLicenseRes.getSfzmhm());
        result.setValidityTime(drivingLicenseRes.getYxqz());
        String status = DateUtil.compare(DateUtil.parseDate(drivingLicenseRes.getYxqz()), new Date()) >= 0 ? "有效" :
                "已过期";
        result.setStatus(status);
        result.setSurplusGrade(String.valueOf(12 - Integer.parseInt(drivingLicenseRes.getLjjf())));

        return ResponseGenerator.success(result);
    }


}



