package com.hikcreate.edl.pub.web.mobile.api.exception;

import com.hikcreate.common.sdk.exception.BusinessException;
import com.hikcreate.common.sdk.response.apiparam.Response;
import com.hikcreate.common.sdk.response.apiparam.ResponseGenerator;
import com.hikcreate.common.sdk.response.statuscode.StatusCode;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.Result;
import com.hikcreate.edl.pub.web.mobile.infra.core.Result.ResultGenerator;
import feign.RetryableException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.List;

/**
 * 全局异常处理类 FOR controller层以下的异常，Filter的异常不能捕获
 */
@Slf4j
@ControllerAdvice
public class GlobalExceptionHandler {

    @ExceptionHandler(value = ServletRequestBindingException.class)
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public Result handleServletRequestBindingException(ServletRequestBindingException e) {
        log.error("param bind failure", e);
        return ResultGenerator.fail(StatusCode.PARAM_ERROR);
    }

    /**
     * 可能原因：@Validated 注解的对象验证不通过
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseBody
    public Result<Void> handle(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String errorMessage = buildErrorMessage(bindingResult);
        log.warn(errorMessage);
        List<FieldError> fieldErrors = bindingResult.getFieldErrors();
        String message = "参数错误";
        if (null != fieldErrors && fieldErrors.size() > 0) {
            FieldError fieldError = fieldErrors.get(0);
            if (null != fieldError) {
                message = fieldError.getDefaultMessage();
            }
        }
        return ResultGenerator.fail(StatusCode.PARAM_ERROR.getCode(), message);
    }

    /**
     * 参数验证失败
     * <p>
     * GET 方法入口
     *
     * @param e
     * @return
     */
    @ExceptionHandler(ConstraintViolationException.class)
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public Result handleValidationException( ConstraintViolationException e) {
        for (ConstraintViolation<?> s : e.getConstraintViolations()) {
            log.error("Request parameter is invalid {}", s.getMessage());
            return ResultGenerator.fail(StatusCode.PARAM_ERROR);
        }
        return ResultGenerator.fail(StatusCode.PARAM_ERROR);
    }

    @ExceptionHandler(value = RetryableException.class)
    @ResponseBody
    public Result retryableExceptionHandler(HttpServletRequest req, RetryableException e) throws Exception {
        log.error(" connect timed out", e);
        return ResultGenerator.fail(StatusCode.SYSTEM_ERROR.getCode(), "服务请求超时或不可用");
    }

    @ExceptionHandler({BusinessException.class})
    @ResponseBody
    public Response<Object> handle(BusinessException e) {
        log.warn(e.toString(), e);
        return ResponseGenerator.fail(e.getCode(), e.getMsg(), e.getErrorMsg(), e.getParam());
    }

    /**
     * 从{@code @BindingResult}中构建异常信息
     * on field 'verifyCode': null];
     *
     * @param bindingResult 绑定结果
     */
    private String buildErrorMessage(BindingResult bindingResult) {
        StringBuilder sb = new StringBuilder("BindException. ");
        sb.append("Field error in object '").append(bindingResult.getObjectName()).append("'. [").append(bindingResult.getTarget()).append("]");
        bindingResult.getFieldErrors()
                .forEach(error -> {
                    sb.append("\r\n on field '").append(error.getField()).append("': ");
                    sb.append("rejected value [").append(error.getRejectedValue()).append("]. ");
                    sb.append("default message [").append(error.getDefaultMessage()).append("]");
                });
        return sb.toString();
    }
}